import os
import sys
from configparser import BasicInterpolation, ConfigParser
from typing import Optional

from pydantic import BaseSettings


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(f"conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    port = config["SERVICE"]["port"]
    host = config["SERVICE"]["host"]


class DBConf:
    DB_URI = config.get('SQL_DB', 'uri')
    if not DB_URI:
        print("Error, environment variable DB_URI not set")
        sys.exit(1)


class Logging:
    level = config.get("LOGGING", "level", fallback="DEBUG")
    level = level or "INFO"
    tb_flag = config.getboolean("LOGGING", "traceback", fallback=True)
    tb_flag = tb_flag if tb_flag is not None else True


class EnvironmentVariables:
    default_link = os.environ.get("DEFAULT_GIT_LINK", default="https://gitlab-pm.knowledgelens.com/")
    git_user_name = os.environ.get("GIT_USERNAME", default="harshavardhan.c")
    git_access_token = os.environ.get("GIT_TOKEN", default="FEMA6PnP63fJCs6DrtZJ")
    config_variables = os.environ.get("CONFIG_MAP_VARIABLES", default="").split(",")
    docker_repo = os.environ.get("DOCKER_REPO",
                                 default="https://gitlab-pm.knowledgelens.com/KnowledgeLens/Products/iLens-2.0/core/devops/docker-compose.git")
    global_configmap = os.environ.get("GLOBAL_VARIABLES_FILE", default="ilens-env-spec-variables.yml")
    helm_repo = os.environ.get("HELM_REPO", default="")


class _ModuleConstants(BaseSettings):
    IOT_DATA_PROCESSOR: Optional[str] = "mqtt-data-processor-worker"
    IOT_DATA_PROCESSOR_MODULE: Optional[str] = "data-processor-worker"


ModuleConstants = _ModuleConstants(_env_file=".env", _env_file_encoding="utf-8")
