if __name__ == "__main__":
    from dotenv import load_dotenv

    load_dotenv()
import argparse
import json
import logging
import os
import shutil
import sys
from typing import List

import git
import gitlab
import jinja2
import ruamel.yaml

from scripts.core import ILensVersionHandler
from scripts.db.psql.databases import get_db_for_func
from scripts.schemas import GetRequest

default_link = "https://gitlab-pm.knowledgelens.com/"
git_user_name = os.environ.get("GIT_USERNAME", default="harshavardhan.c")
git_access_token = os.environ.get("GIT_TOKEN", default="FEMA6PnP63fJCs6DrtZJ")
config_variables = os.environ.get("CONFIG_MAP_VARIABLES", default="").split(",")
helm_repo = os.environ.get("HELM_REPO", default="")
global_configmap = os.environ.get("GLOBAL_VARIABLES_FILE", default="")

HELM_PATH = "/ilens-core/ilens-modules"
HELM_STORE_PATH = "./helm-charts"
REPO_CLONE_PATH = "./data"


def render_helm_chart(helm_name, data_dict, variables_list, helm_template_file):
    try:
        output_path = "helm-charts"
        if not os.path.exists(output_path):
            os.makedirs(output_path)
        helm_path = os.path.join(output_path, f'{helm_name}.yml')

        environment = jinja2.Environment(
            loader=jinja2.FileSystemLoader(searchpath='./templates'),
            trim_blocks=True,
            variable_start_string='<{', variable_end_string='}>')
        _render = environment.get_template(helm_template_file).render(**data_dict,
                                                                      variables=variables_list)
        with open(helm_path, "w") as fp:
            fp.write(_render)

    except Exception as e:
        logging.exception(f"Exception occurred while rendering the helm file {helm_name} - {e.args}")


def clone_repository(repo_link, module_output_path, clone_branch):
    try:
        if repo_link.split("https://")[-1].startswith("gitlab-pm"):
            repo_link = repo_link.replace("https://", f"https://{git_user_name}:{git_access_token}@")
        repo = git.Repo.clone_from(repo_link, module_output_path,
                                   branch=clone_branch)
        return True
    except Exception as e:
        logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
        return False
    
def get_module_names_from_files():
    try:
        module_names = set()
        dir_list = os.listdir(f'{REPO_CLONE_PATH}{HELM_PATH}')
        for file in dir_list:
            if file.endswith(".yml"):
                module_names.add(file.removesuffix(".yml"))
        return list(module_names)
    except Exception as e:
        logging.exception(f"Exception occurred while fetching module names from files: {e.args}")


def clone_repository_with_defined_file(repo_link: str, file_output_path, clone_branch, private_token, clone_file_path):
    try:
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=private_token)
        search_str = repo_link_split[-1].replace(".git", "")
        pl = gl.projects.list(search=search_str)
        if not pl:
            return False
        pl = pl[0]
        with open(file_output_path, 'wb') as f:
            pl.files.raw(file_path=clone_file_path, ref=clone_branch, streamed=True, action=f.write)
        return True
    except Exception as e:
        logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
        return False


def convert_yaml_to_json(yaml_file_path):
    try:
        if not os.path.exists(yaml_file_path):
            return {}
        _yaml = ruamel.yaml.YAML(typ='safe')
        with open(yaml_file_path) as fpi:
            yaml_dict = _yaml.load(fpi)
        return yaml_dict
    except Exception as e:
        logging.exception(f"Exception Occurred while reading the yaml file {e.args}")
        return {}
    
def push_helm_deployments(repo: git.Repo):
    try:
        files_list = os.listdir(HELM_STORE_PATH)
        for file in files_list:
            shutil.copy(f'{HELM_STORE_PATH}/{file}', f'{REPO_CLONE_PATH}{HELM_PATH}/{file}')
        repo.git.add("-A")
        repo.git.add(update=True)
        repo.index.commit("helm files update")
        repo.remotes.origin.push()
    except Exception as e:
        logging.exception(f'Exception while pushing helm deployments: {e.args}')
        
def pull_global_config(repo_link: str, branch: str, file_name: str):
    if repo_link.split("https://")[-1].startswith("gitlab-pm"):
        repo_link = repo_link.replace("https://", f"https://{git_user_name}:{git_access_token}@")
    repo = git.Repo.clone_from(url=f'{repo_link}.git', to_path=REPO_CLONE_PATH)
    repo.git.checkout(branch)
    shutil.copy(f'{REPO_CLONE_PATH}{HELM_PATH}/{file_name}', ".")
    return repo

def remove_all_files_from_repo(exclude_file: list):
    dir_list = os.listdir(f'{REPO_CLONE_PATH}{HELM_PATH}')
    remove_list = list(set(dir_list).difference(exclude_file))
    for file in remove_list:
        if os.path.isfile(f'{REPO_CLONE_PATH}{HELM_PATH}/{file}'):
            os.remove(f'{REPO_CLONE_PATH}{HELM_PATH}/{file}')
    return True


ap = argparse.ArgumentParser()
db_handler = ILensVersionHandler()
if __name__ == '__main__':
    ap.add_argument(
        "--ilens_version",
        "-iv",
        required=False,
        default=None,
        help="ILens Version Tag",
    )
    ap.add_argument(
        "--release_version",
        "-rv",
        required=False,
        default=None,
        help="ILens Release Tag",
    )
    ap.add_argument(
        "--client_name",
        "-cn",
        required=False,
        default=None,
        help="Client Name Tag"
    )
    
    arguments = vars(ap.parse_args())
    _ilens_version = arguments["ilens_version"]
    _release_version = arguments["release_version"]
    _client_name = arguments['client_name']
    if not _ilens_version or not _release_version or not _client_name or not global_configmap:
        print("global_configmap, client_name, ilens_version and release_version details not found!!!!!")
        sys.exit()
    _branch = f"{_client_name}_{_ilens_version}.{_release_version}"
    try:
        if not (helm_push_repo:=pull_global_config(repo_link=helm_repo, branch=_branch, file_name=global_configmap)):
            logging.error(f"Cannot clone helm repo with branch: {_branch}")
            sys.exit()
        with open("config.json", "r") as f:
            data = json.load(f)
        _module_names = list(set(get_module_names_from_files()).difference(list(global_configmap.removesuffix(".yml"))))
        data['git_modules'] = [x for x in data['git_modules'] if (x['module_name'] in _module_names)]
        global_config_data = convert_yaml_to_json(global_configmap)
        for _data in data.get('git_modules'):
            _ilens_version = _data.get("ilens_version", _ilens_version)
            _release_version = _data.get("ilens_version", _release_version)
            git_link = _data["git_link"]
            branch = _data["branch"]
            variables_file = _data.get("variables_file") or "variables.yml"

            module_path = os.path.join("tmp")

            module_name = _data.get('module_name') or git_link.split("/")[-1].split(".git")
            module_path = os.path.join(module_path, module_name)
            if not os.path.exists(module_path):
                os.makedirs(module_path)
            variables_file_path = os.path.join(module_path, variables_file)
            if not clone_repository_with_defined_file(repo_link=git_link, clone_branch=branch,
                                                    file_output_path=variables_file_path,
                                                    private_token=git_access_token, clone_file_path=variables_file):
                logging.debug("Failed to clone module!! Skipping Helm File Preparation")
                continue
            _module_data = convert_yaml_to_json(variables_file_path)
            global_config_vars = global_config_data['data']
            env_variables_from_yml = _module_data.get('deployment', {}).get('environmentVar', [])
            env_variables_from_yml = {_v['name']:_v['value'] for _v in env_variables_from_yml if
                            {'name', 'value'}.issubset(set(list(_v.keys())))}
            env_variables_from_yml |= global_config_vars
            template_file = _data.get("template_file") or "helm_service_deployment.yaml"
            session_obj = get_db_for_func()
            module_info = db_handler.get_module_versions(
                input_data=GetRequest(module_name=module_name, client='iLens', ilens_version=_ilens_version,
                                    release_version=_release_version), db=session_obj)
            session_obj.close()
            _data["image_tag"] = module_info.get("image_tag", "-")
            render_helm_chart(helm_name=module_name, data_dict=_data, variables_list=env_variables_from_yml,
                            helm_template_file=template_file)
        push_helm_deployments(helm_push_repo)
    except Exception as e:
        logging.exception(f'Unable to update helms: {e.args}')
    finally:
        if os.path.exists(REPO_CLONE_PATH):
            shutil.rmtree(REPO_CLONE_PATH)
        if os.path.exists("./tmp"):
            shutil.rmtree("./tmp")
        if os.path.exists(HELM_STORE_PATH):
            shutil.rmtree(HELM_STORE_PATH)
        if os.path.isfile(f"./{global_configmap}"):
            os.remove(f"./{global_configmap}")