if __name__ == "__main__":
    from dotenv import load_dotenv

    load_dotenv()
import argparse
import json
import logging
import os
import shutil
import sys

import git
import gitlab
import jinja2
import ruamel.yaml

from scripts.core import ILensVersionHandler
from scripts.db.psql.databases import get_db_for_func
from scripts.schemas import GetRequest

git_user_name = os.environ.get("GIT_USERNAME", default="harshavardhan.c")
git_access_token = os.environ.get("GIT_TOKEN", default="FEMA6PnP63fJCs6DrtZJ")
config_variables = os.environ.get("CONFIG_MAP_VARIABLES", default="").split(",")
helm_repo = os.environ.get("HELM_REPO", default="")

HELM_PATH = "/ilens-core/ilens-modules"
HELM_STORE_PATH = "./helm-charts"


def render_helm_chart(helm_name, data_dict, variables_list, helm_template_file):
    try:
        output_path = "helm-charts"
        if not os.path.exists(output_path):
            os.makedirs(output_path)
        helm_path = os.path.join(output_path, f'{helm_name}.yaml')

        environment = jinja2.Environment(
            loader=jinja2.FileSystemLoader(searchpath='./templates'),
            trim_blocks=True,
            variable_start_string='<{', variable_end_string='}>')
        _render = environment.get_template(helm_template_file).render(module_name=helm_name, **data_dict,
                                                                      variables=variables_list)
        with open(helm_path, "w") as fp:
            fp.write(_render)

    except Exception as e:
        logging.exception(f"Exception occurred while rendering the helm file {helm_name} - {e.args}")


def clone_repository(repo_link, module_output_path, clone_branch):
    try:
        if repo_link.split("https://")[-1].startswith("gitlab-pm"):
            repo_link = repo_link.replace("https://", f"https://{git_user_name}:{git_access_token}@")
        repo = git.Repo.clone_from(repo_link, module_output_path,
                                   branch=clone_branch)
        return True
    except Exception as e:
        logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
        return False


def clone_repository_with_defined_file(repo_link: str, file_output_path, clone_branch, private_token, clone_file_path):
    try:
        base_url = os.environ.get("GIT_BASE_URL", default="https://gitlab-pm.knowledgelens.com/")
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=private_token)
        search_str = repo_link_split[-1].replace(".git", "")
        pl = gl.projects.list(search=search_str)
        if not pl:
            return False
        pl = pl[0]
        with open(file_output_path, 'wb') as f:
            pl.files.raw(file_path=clone_file_path, ref=clone_branch, streamed=True, action=f.write)
        return True
    except Exception as e:
        logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
        return False


def convert_yaml_to_json(yaml_file_path):
    try:
        if not os.path.exists(yaml_file_path):
            return {}
        _yaml = ruamel.yaml.YAML(typ='safe')
        with open(yaml_file_path) as fpi:
            yaml_dict = _yaml.load(fpi)
        return yaml_dict
    except Exception as e:
        logging.exception(f"Exception Occurred while reading the yaml file {e.args}")
        return {}
    
def push_helm_deployments(repo_link:str, private_token:str, ilens_version, release_version, client_name):
    try:
        base_url = os.environ.get("GIT_BASE_URL", default="https://gitlab-pm.knowledgelens.com/")
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=private_token)
        search_str = repo_link_split[-1].replace(".git", "")
        pl = gl.projects.list(search=search_str)
        if not pl:
            return False
        pl = pl[0]
        pl.branches.create({ 'branch': f"{client_name}_{ilens_version}.{release_version}", 'ref': pl.default_branch})
        commit_data = {
            'branch': f"{client_name}_{ilens_version}.{release_version}",
            'commit_message': f"{client_name}_{ilens_version}.{release_version} helm creation"
        }
        commit_actions = []
        files_list = os.listdir(HELM_STORE_PATH)
        for file in files_list:
            _action = {
                'action': 'create',
                'file_path': f'{HELM_PATH}/{file}',
                'content': open(f'{HELM_STORE_PATH}/{file}').read()
            }
            commit_actions.append(_action)
        commit_data |= {'actions': commit_actions}
        pl.commits.create(commit_data)
    except Exception as e:
        logging.exception(f'Exception while pushing helm deployments: {e.args}')


ap = argparse.ArgumentParser()
db_handler = ILensVersionHandler()
if __name__ == '__main__':
    ap.add_argument(
        "--ilens_version",
        "-iv",
        required=False,
        default=None,
        help="ILens Version Tag",
    )
    ap.add_argument(
        "--release_version",
        "-rv",
        required=False,
        default=None,
        help="ILens Release Tag",
    )
    ap.add_argument(
        "--client_name",
        "-cn",
        required=False,
        default=None,
        help="Client Name Tag"
    )
    ap.add_argument(
        "--git_repos",
        "-gr",
        required=False,
        default=None,
        help="Git repos to be added in helm",
        nargs="+"
    )
    arguments = vars(ap.parse_args())
    _ilens_version = arguments["ilens_version"]
    _release_version = arguments["release_version"]
    _client_name = arguments['client_name']
    _git_repos = arguments["git_repos"]
    if not _ilens_version or not _release_version or not _client_name or not _git_repos:
        print("git_repos, client_name, ilens_version and release_version details not found!!!!!")
        sys.exit()
    with open("config.json", "r") as f:
        data = json.load(f)
    data['git_modules'] = [x for x in data['git_modules'] if x['git_link'] in _git_repos]
    for _data in data.get('git_modules'):
        _ilens_version = _data.get("ilens_version", _ilens_version)
        _release_version = _data.get("ilens_version", _release_version)
        git_link = _data["git_link"]
        branch = _data["branch"]
        variables_file = _data.get("variables_file") or "variables.yml"

        module_path = os.path.join("tmp")

        module_name = git_link.split("/")[-1].split(".git")[0]
        module_path = os.path.join(module_path, module_name)
        if not os.path.exists(module_path):
            os.makedirs(module_path)
        variables_file_path = os.path.join(module_path, variables_file)
        if not clone_repository_with_defined_file(repo_link=git_link, clone_branch=branch,
                                                  file_output_path=variables_file_path,
                                                  private_token=git_access_token, clone_file_path=variables_file):
            logging.debug("Failed to clone module!! Skipping Helm File Preparation")
            continue
        _module_data = convert_yaml_to_json(variables_file_path)
        env_variables_from_yml = _module_data.get('deployment', {}).get('environmentVar', [])
        env_variables_from_yml = [_v for _v in env_variables_from_yml if
                          {'name', 'value'}.issubset(set(list(_v.keys())))]
        template_file = _data.get("template_file") or "helm_service_deployment.yaml"
        session_obj = get_db_for_func()
        module_info = db_handler.get_module_versions(
            input_data=GetRequest(module_name=module_name, client='iLens', ilens_version=_ilens_version,
                                  release_version=_release_version), db=session_obj)
        session_obj.close()
        _data["image_tag"] = module_info.get("image_tag", "-")
        render_helm_chart(helm_name=module_name, data_dict=_data, variables_list=env_variables_from_yml,
                          helm_template_file=template_file)
    push_helm_deployments(helm_repo, git_access_token, _ilens_version, _release_version, _client_name)
    shutil.rmtree("./tmp")
    shutil.rmtree(HELM_STORE_PATH)