import os

from scripts.config import EnvironmentVariables
from scripts.core import ILensVersionHandler
from scripts.core.git_handler import GitHandler
from scripts.logging import logging
from scripts.schemas import DockerComposeSchema
from scripts.utils.common_utils import CommonUtils


class DockerHandler:

    def __init__(self, git_user_name, git_access_token):
        self.git_handler_obj = GitHandler(user_name=git_user_name, access_token=git_access_token)
        self.common_util = CommonUtils()
        self.db_handler = ILensVersionHandler()

    def process_module_request(self, compose_data: dict, variables_file, general_path,
                               **kwargs):
        try:
            ilens_version = kwargs.get("ilens_version")
            branch_name = kwargs.get("branch_name")
            release_version = kwargs.get("release_version")
            for _module, module_dict in compose_data['services'].items():
                variable_name = f'x-{_module}-image'
                module_path = os.path.join(general_path)
                module_path = os.path.join(module_path, _module)
                if not os.path.exists(module_path):
                    os.makedirs(module_path)
                variables_file_path = os.path.join(module_path, variables_file)
                git_info = self.git_handler_obj.get_git_url_by_module_name(module_name=_module)
                if not git_info:
                    logging.debug("Failed to fetch module info!! Skipping Helm File Preparation")
                    continue
                if not self.git_handler_obj.clone_repository_with_defined_file(repo_link=git_info,
                                                                               clone_branch=branch_name,
                                                                               file_output_path=variables_file_path,
                                                                               clone_file_path=variables_file):
                    logging.debug("Failed to clone module!! Skipping Docker File Preparation")
                    continue
                _module_data = self.common_util.convert_yaml_to_define_obj(variables_file_path)
                # session_obj = get_db_for_func()
                # module_info = self.db_handler.get_module_versions(
                #     input_data=GetRequest(module_name=_module, client='iLens', ilens_version=ilens_version,
                #                           release_version=release_version), db=session_obj)
                # session_obj.close()
                _module_data = self.common_util.convert_yaml_to_define_obj(variables_file_path)
                module_info = {}
                module_env_variables = _module_data.get('deployment', {}).get('environmentVar', [])
                module_env_variables = {_v['name']: _v.get('value') for _v in module_env_variables}
                image_url = module_info.get("image_url", '') if module_info else ''
                if image_url:
                    module_dict['image'] = image_url
                    compose_data[variable_name] = image_url
                existing_env_variables = module_dict.get('environment', {})
                diff_keys = list(
                    set(existing_env_variables.keys()).symmetric_difference(set(module_env_variables.keys())))
                for _each in diff_keys:
                    value = module_env_variables.get(_each)
                    if _each.lower() in {'port', 'service_port', 'module_port'} or not value:
                        continue
                    if value.startswith('{{') and value.endswith('}}'):
                        value = f"${{{value.lstrip('{{').rstrip('}}').rstrip(' ').lstrip(' ')}}}"
                    module_dict['environment'].update({_each: value})
            return compose_data

        except Exception as e:
            logging.exception(f'Exception occurred while process each module {e.args}')
            return False

    def process_compose_data_for_existing_files(self, docker_compose_path, variables_file, arguments,
                                                tmp_path, output_path, source_branch, destination_branch):
        docker_repo = EnvironmentVariables.docker_repo
        global_configmap = EnvironmentVariables.global_configmap

        try:
            files_info = os.listdir(docker_compose_path)
            sorted_files = list(filter(lambda f: f.endswith(".yml") and "docker" in f, files_info))
            global_config_data = self.common_util.convert_yaml_to_define_obj(
                os.path.join(docker_compose_path, global_configmap))

            for _file in sorted_files:
                docker_compose_data = self.common_util.convert_yaml_to_define_obj(
                    os.path.join(docker_compose_path, _file),
                    load_type=None)
                service_dict = docker_compose_data.get("services")
                if not service_dict:
                    logging.debug(f'Services not found for current docker compose file  - {_file}')
                if response_data := self.process_module_request(compose_data=docker_compose_data,
                                                                variables_file=variables_file,
                                                                general_path=tmp_path, **arguments):
                    compose_out_file_path = os.path.join(output_path, _file)
                    self.common_util.convert_json_to_yaml(json_data=response_data,
                                                          output_file_path=compose_out_file_path)
            self.git_handler_obj.push_deployments_to_git(repo_link=docker_repo,
                                                         private_token=EnvironmentVariables.git_access_token,
                                                         branch=source_branch,
                                                         folder_path=output_path, helm_deployment=False,
                                                         base_path=docker_compose_path)
            self.git_handler_obj.create_merge_request(repo_link=docker_repo, source_branch=source_branch,
                                                      destination_branch=destination_branch)

        except Exception as e:
            logging.exception(f'Exception Occurred while processing the compose data info {e.args}')

    def process_compose_data_by_template(self, modules, compose_info: dict, variables_file, arguments,
                                         tmp_path, output_path):
        compose_dict = {"services": {}}
        try:
            for module in modules:
                compose_dict['services'].update({
                    module: DockerComposeSchema().dict()
                })
            compose_out_file_path = os.path.join(output_path, "docker-compose.yml")
            jinja_template_file = 'docker_deployment.yml'
            template_path = "./templates"
            compose_data = self.process_module_request(compose_data=compose_dict,
                                                       variables_file=variables_file,
                                                       general_path=tmp_path, **arguments)
            self.common_util.render_helm_chart(data_dict={"modules": compose_data.get('services', {})},
                                               jinja_template_file=jinja_template_file, template_path=template_path,
                                               outfile_path=compose_out_file_path)

        except Exception as e:
            logging.exception(f'Exception Occurred while processing the compose data by template {e.args}')
