if __name__ == "__main__":
    from dotenv import load_dotenv

    load_dotenv()
import argparse
import logging
import os
import shutil
import sys
import time

import gitlab

from scripts.core import ILensVersionHandler
from scripts.core.git_handler import GitHandler
from scripts.core.helm_handler import HelmHandler
from scripts.db.psql.databases import get_db_for_func
from scripts.schemas import GetRequest
from scripts.utils.common_utils import CommonUtils

default_link = "https://gitlab-pm.knowledgelens.com/"
git_user_name = os.environ.get("GIT_USERNAME", default="harshavardhan.c")
git_access_token = os.environ.get("GIT_TOKEN", default="FEMA6PnP63fJCs6DrtZJ")
config_variables = os.environ.get("CONFIG_MAP_VARIABLES", default="").split(",")
helm_repo = os.environ.get("HELM_REPO", default="")
global_configmap = os.environ.get("GLOBAL_VARIABLES_FILE", default="ilens-global-configmap.yml")

HELM_PATH = "/ilens-core/ilens-modules"
HELM_STORE_PATH = "./helm-charts"

git_handler_obj = GitHandler(user_name=git_user_name, access_token=git_access_token)


def push_helm_deployments(repo_link: str, private_token: str, branch: str, final_helm_path, base_path: str):
    try:
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=private_token)
        search_str = repo_link_split[-1].replace(".git", "")
        pl = gl.projects.list(search=search_str)
        if not pl:
            return False
        pl = pl[0]
        commit_actions = []
        files_list = os.listdir(final_helm_path)
        branches = pl.branches.list(get_all=True)
        branches_names = [x.name for x in branches]
        if branch not in branches_names:
            pl.branches.create({'branch': branch,
                                'ref': 'master' if branch.split("_")[0] not in branches_names else branch.split("_")[
                                    0]})
        if not files_list:
            logging.debug('Files not found for pushing to git.')
        for file in files_list:
            _action = {
                'action': 'update',
                'file_path': os.path.join(HELM_PATH, file),
                'content': open(f'{final_helm_path}/{file}').read()
            }
            commit_actions.append(_action)
        commit_data = {'branch': branch, 'commit_message': f"{branch} helm creation"} | {'actions': commit_actions}

        pl.commits.create(commit_data)
    except Exception as e:
        logging.exception(f'Exception while pushing helm deployments: {e.args}')


ap = argparse.ArgumentParser()
db_handler = ILensVersionHandler()
common_util = CommonUtils()
helm_handler = HelmHandler()
if __name__ == '__main__':
    ap.add_argument(
        "--ilens_version",
        "-iv",
        required=False,
        default=None,
        help="ILens Version Tag",
    )
    ap.add_argument(
        "--release_version",
        "-rv",
        required=False,
        default=None,
        help="ILens Release Tag",
    )
    ap.add_argument(
        "--client_name",
        "-cn",
        required=False,
        default=None,
        help="Client Name Tag"
    )
    ap.add_argument(
        "--branch_name",
        "-b",
        required=False,
        default=None,
        help="Branch Name"
    )
    ap.add_argument(
        "--git_repos",
        "-gr",
        required=False,
        default=None,
        help="Git repos to be added in helm",
        nargs="+"
    )
    ap.add_argument(
        "--module_names",
        "-mn",
        required=False,
        default=None,
        help="Module names to be added in helm",
        nargs="+"
    )
    HELM_TEMP_PATH = f"{int(time.time())}_helm_tmp_path"
    GENERAL_TEMP_PATH = f"{int(time.time())}_tmp"
    OUTPUT_PATH = f"{int(time.time())}_helm-charts"
    try:
        arguments = vars(ap.parse_args())
        _ilens_version = arguments["ilens_version"]
        _release_version = arguments["release_version"]
        _client_name = arguments['client_name']
        _git_repos = arguments["git_repos"]
        _module_names = arguments["module_names"]
        _branch_name = arguments['branch_name'] or "master"
        if not _ilens_version or not _release_version or not _client_name or not (
                _git_repos or _module_names) or not global_configmap:
            print(
                "global_configmap, git_repos, module_names, client_name, ilens_version and release_version details not found!!!!!")
            sys.exit()
        _branch = f"{_client_name}_{_ilens_version}.{_release_version}"
        if not os.path.exists(HELM_TEMP_PATH):
            os.makedirs(HELM_TEMP_PATH)
        if not os.path.exists(OUTPUT_PATH):
            os.makedirs(OUTPUT_PATH)
        helm_path = os.path.join(HELM_TEMP_PATH, "helm-charts")
        if not git_handler_obj.clone_repository(repo_link=helm_repo, module_output_path=helm_path,
                                                clone_branch=_client_name):
            logging.error(f"Cannot clone helm repo with branch: {_branch}")
            sys.exit()
        base_helm_directory_path = os.path.join(HELM_TEMP_PATH, "helm-charts", "ilens-core", "ilens-modules")
        # global_config_data = convert_yaml_to_json(os.path.join(base_helm_directory_path, global_configmap))
        global_config_data = common_util.convert_yaml_to_json("ilens-global-configmap.yml")

        variables_file = "variables.yml"
        template_path = os.path.join(GENERAL_TEMP_PATH, "templates")
        if not os.path.exists(template_path):
            os.makedirs(template_path)
        for _module in _module_names:
            module_path = os.path.join(GENERAL_TEMP_PATH)
            module_path = os.path.join(module_path, _module)
            if not os.path.exists(module_path):
                os.makedirs(module_path)
            variables_file_path = os.path.join(module_path, variables_file)
            git_info = git_handler_obj.get_git_url_by_module_name(module_name=_module)
            if not git_info:
                logging.debug("Failed to fetch module info!! Skipping Helm File Preparation")
                continue
            if not git_handler_obj.clone_repository_with_defined_file(repo_link=git_info, clone_branch=_branch_name,
                                                                      file_output_path=variables_file_path,
                                                                      clone_file_path=variables_file):
                logging.debug("Failed to clone module!! Skipping Helm File Preparation")
                continue
            _module_data = common_util.convert_yaml_to_json(variables_file_path)
            module_env_variables = _module_data.get('deployment', {}).get('environmentVar', [])
            module_env_variables = {_v['name']: _v for _v in module_env_variables}
            template_file = os.path.join(template_path, f'{_module}.yml')
            helm_out_file_path = os.path.join(OUTPUT_PATH, f'{_module}.yml')
            session_obj = get_db_for_func()
            module_info = db_handler.get_module_versions(
                input_data=GetRequest(module_name=_module, client='iLens', ilens_version=_ilens_version,
                                      release_version=_release_version), db=session_obj)
            session_obj.close()
            image_url = module_info.get("image_url")
            existing_yml_path = os.path.join(base_helm_directory_path, f'{_module}.yml')
            helm_handler.create_helm_deployment_file(template_yml_path=existing_yml_path, image_tag=image_url,
                                                     module_env_variables=module_env_variables,
                                                     template_file=template_file, template_path=template_path,
                                                     helm_out_file_path=helm_out_file_path,
                                                     global_config_data=global_config_data, module_name=_module)
            files_info = os.listdir(base_helm_directory_path)
            sorted_files = list(filter(lambda f: f.startswith(_module), files_info))
            updated_files_info = [x for x in sorted_files if f'{_module}.yml' != x]
            for _file in updated_files_info:
                sub_file_path = os.path.join(base_helm_directory_path, _file)
                helm_out_file_path = os.path.join(OUTPUT_PATH, _file)
                template_file = os.path.join(template_path, _file)
                helm_handler.create_helm_deployment_file(template_yml_path=sub_file_path, image_tag=image_url,
                                                         module_env_variables=module_env_variables,
                                                         template_file=template_file, template_path=template_path,
                                                         helm_out_file_path=helm_out_file_path,
                                                         global_config_data=global_config_data,
                                                         module_name=_file.split(".yml")[0])
        push_helm_deployments(helm_repo, git_access_token, _branch, final_helm_path=OUTPUT_PATH, base_path=helm_path)

    except Exception as e:
        logging.exception(f"Exception Occurred while processing the Helm-Script Preparation {e.args}")
    finally:
        # shutil.rmtree(HELM_STORE_PATH)
        shutil.rmtree(HELM_TEMP_PATH)
        shutil.rmtree(GENERAL_TEMP_PATH)
