import json
from datetime import datetime

import pytz
from sqlalchemy.orm import Session

from scripts.constants import ResponseCodes
from scripts.core.engine.oee_calculator import OEEEngine
from scripts.db.psql.oee_discrete import DiscreteOEE
from scripts.db.redis_connections import oee_production_db
from scripts.logging import logger
from scripts.schemas.batch_oee import OEEDataInsertRequest, BatchOEEData, OEEDataSaveRequest
from scripts.schemas.response_models import DefaultResponse
from scripts.utils.common_utils import CommonUtils

oee_engine = OEEEngine()


class CalculateBatchOEEHandler:

    def __init__(self, project_id=None):
        self.common_util = CommonUtils()

    def calculate_oee(self, db, request_data: OEEDataInsertRequest):
        table_obj = DiscreteOEE(db=db)
        try:
            record_presence = table_obj.get_oee_data_by_reference_id(reference_id=request_data.reference_id,
                                                                     hierarchy=request_data.hierarchy,
                                                                     project_id=request_data.project_id)

            redis_key = f"{request_data.project_id}${request_data.reference_id}"
            if not record_presence:
                if not request_data.prod_start_time:
                    request_data.prod_start_time = datetime.now().astimezone(
                        tz=pytz.timezone(request_data.tz)).isoformat()
                request_data = OEEDataSaveRequest(**request_data.dict(exclude_none=True))
                request_data.downtime = self.common_util.get_downtime_details_by_hierarchy(
                    hierarchy=request_data.hierarchy, project_id=request_data.project_id)
                oee_calculation = oee_engine.start_batch_oee_calc(request_data=request_data)
                self.save_oee_data(oee_calculation, db)
                oee_production_db.set(name=redis_key,
                                      value=json.dumps(BatchOEEData(**request_data.dict(exclude_none=True))))
                response = DefaultResponse(
                    status=ResponseCodes.SUCCESS,
                    data=oee_calculation,
                    message="OEE saved Successfully",
                )
                return response
            status = self.update_oee_data(oee_data=BatchOEEData(**request_data.dict(exclude_none=True)),
                                          old_record=record_presence, db=db)
            if status:
                if request_data.prod_end_time:
                    oee_production_db.delete(redis_key)
                else:
                    oee_production_db.set(name=redis_key,
                                          value=json.dumps(record_presence))
            response = DefaultResponse(
                status=ResponseCodes.SUCCESS,
                data=status,
                message="OEE updated Successfully",
            )
            return response
        except Exception as e:
            logger.exception(f"Exception while saving oee record: {e}")
            raise e

    @staticmethod
    def save_oee_data(oee_data: BatchOEEData, db: Session):
        table_obj = DiscreteOEE(db=db)
        try:
            table_obj.insert_one(table=table_obj.table, insert_json=oee_data.dict())
            return True
        except Exception as e:
            raise e

    @staticmethod
    def update_oee_data(oee_data: BatchOEEData, old_record: dict, db: Session):
        table_obj = DiscreteOEE(db=db)
        try:
            old_record.update(**oee_data.dict(exclude_none=True))
            oee_calculation = oee_engine.start_batch_oee_calc(
                request_data=OEEDataSaveRequest(**old_record))
            table_obj.update(update_json=oee_calculation.dict(), table=table_obj.table, filters=[{'expression': 'eq',
                                                                                                  'column': table_obj.table.project_id,
                                                                                                  'value': oee_data.project_id
                                                                                                  },
                                                                                                 {'expression': 'eq',
                                                                                                  'column': table_obj.table.reference_id,
                                                                                                  'value': oee_data.reference_id
                                                                                                  },
                                                                                                 {'expression': 'eq',
                                                                                                  'column': table_obj.table.hierarchy,
                                                                                                  'value': oee_data.hierarchy
                                                                                                  }], update_one=True)
            return True
        except Exception as e:
            raise e
