from fastapi import HTTPException, Request, Response, status
from fastapi.openapi.models import APIKey, APIKeyIn
from fastapi.security import APIKeyCookie
from fastapi.security.api_key import APIKeyBase
from pydantic import BaseModel, Field

from scripts.config import Services
from scripts.database.redis.redis_conn import login_db
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.utils.security.jwt_util import JWT


class MetaInfoSchema(BaseModel):
    project_id: str = ""
    user_id: str = ""
    ip_address: str = ""
    login_type: str = ""
    login_token: str = Field(alias="login-token")

    class Config:
        allow_population_by_field_name = True


class _CookieAuthentication(APIKeyBase):
    """
    Authentication backend using a cookie.
    Internally, uses a JWT token to store the data.
    """

    scheme: APIKeyCookie
    cookie_name: str
    cookie_secure: bool

    def __init__(
            self,
            cookie_name: str = "login-token",
    ):
        super().__init__()
        self.model: APIKey = APIKey(**{"in": APIKeyIn.cookie}, name=cookie_name)
        self.scheme_name = self.__class__.__name__
        self.cookie_name = cookie_name
        self.scheme = APIKeyCookie(name=self.cookie_name, auto_error=False)
        self.login_redis = login_db
        self.jwt = JWT()

    def __call__(self, request: Request, response: Response) -> MetaInfoSchema:
        cookies = request.cookies
        login_token = cookies.get(self.cookie_name) or request.headers.get(
            self.cookie_name
        )
        if not login_token or login_token != Services.PROJECT_NAME:
            raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED)

        if login_token == Services.PROJECT_NAME:
            return MetaInfoSchema(
                project_id=Services.PROJECT_ID,
                ip_address=request.client.host,  # type: ignore
                login_token=cookies.get("login-token"),
            )
        jwt_token = self.login_redis.get(login_token)
        if not jwt_token:
            raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED)

        try:
            decoded_token = self.jwt.validate(token=jwt_token)
            if not decoded_token:
                raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED)
        except Exception as e:
            logger.exception(e)
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail=ErrorMessages.UNKNOWN_ERROR,
            )

        user_id = decoded_token.get("user_id")
        project_id = decoded_token.get("project_id")
        if not user_id or not project_id:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Token doesn't have required fields",
            )

        return MetaInfoSchema(
            project_id=project_id,
            user_id=user_id,
            ip_address=request.client.host,  # type: ignore
            login_token=cookies.get("login-token"),
        )


CookieAuthentication = auth = _CookieAuthentication()
