import smtplib
import ssl
from datetime import timedelta, datetime
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

from fastapi import status
from fastapi.responses import JSONResponse

from scripts.config import Services, Secrets
from scripts.core.handlers.normal_login import NormalLogin
from scripts.database.mongo.mongo_db import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.schemas.default_responses import DefaultFailureResponse, DefaultResponse, DefaultSuccessResponse
from scripts.utils.security.jwt_util import JWT
from scripts.utils.security.password_util import EncryptDecryptPassword

obj_mongo_user = MongoUser()
jwt = JWT()


class LoginHandlers:
    def __init__(self):
        self.obj_login_handler = NormalLogin()
        self.pass_decrypt = EncryptDecryptPassword()
        self.login_type = ""

    def normal_login(self, user_data, request):
        self.login_type = "general_login"
        # decrypting the password from the UI
        decrypted_password = self.pass_decrypt.password_decrypt(user_data.password)
        # validating the received inputs empty or not
        # password decrypted form - token "password"
        responses = self.obj_login_handler.user_data_validation(
            user_data.email,
            decrypted_password.split("\"")[1])
        # Account is not registered
        if responses is not None:
            return JSONResponse(content=DefaultFailureResponse(status="failed",
                                                               message=responses).dict(),
                                status_code=status.HTTP_200_OK)
        # checking for the account and password matching
        user_data_response, data = self.obj_login_handler.db_password_matching(self.login_type, user_data,
                                                                               decrypted_password.split("\"")[1])
        # if the passwords doesn't match with the db data
        if user_data_response is not None:
            return JSONResponse(content=DefaultFailureResponse(status="failed",
                                                               message=data).dict(),
                                status_code=status.HTTP_200_OK)
        #  generating the access tokens
        responses, exp = self.obj_login_handler.generate_cookie_tokens(data, request)
        # token generation unsuccessful
        if responses is None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_TOKEN_GENERATION).dict(),
                status_code=status.HTTP_200_OK)
        # sending successful response to UI
        response = JSONResponse(
            content=DefaultResponse(status="success", message="Logged In Successfully",
                                    data=data).dict(),
            status_code=status.HTTP_200_OK, headers={"Content-Type": "application/json"})
        response.set_cookie(key="login-token", value=responses, expires=exp)
        return response

    # v2
    def google_login(self, request):
        pass

    # v2
    def microsoft_login(self, request):
        pass

    # forgot password handler
    @staticmethod
    def forgot_password_handler(email):
        try:
            # Check if user exists in database
            # If user exists, send forgot password email with JWT token
            # This should include email and expire time
            # Send email using MIME
            db_user_data = obj_mongo_user.fetch_one_user_details({"email": email})
            # if the user is not available
            if not db_user_data:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_200_OK)
            mail = MIMEMultipart()
            mail['From'] = Services.EMAIL_SENDER
            mail['To'] = email
            mail['Subject'] = "Link TO Reset Password"
            to_encode = {"email": email}
            expire = datetime.utcnow() + timedelta(minutes=Secrets.TOKEN_EXPIRE_TIME)
            to_encode.update({"exp": expire})
            jwt_token = jwt.encode(to_encode)
            html = ''
            # Load the HTML file
            try:
                with open(Services.HTML_LINK, "r") as f:
                    html = f.read()
                html = html.replace("{{ message }}", "Please click the link to reset your password:").replace(
                    "{{ link }}", Services.RESET_ENDPOINT + "=" + str(jwt_token))
            except Exception as e:
                logger.exception(e)
            html_body = MIMEText(html, "html")
            mail.attach(html_body)
            context = ssl.create_default_context()
            with smtplib.SMTP_SSL(Services.EMAIL_SMTP, Services.EMAIL_PORT, context=context) as smtp:
                smtp.login(Services.EMAIL_SENDER, Services.EMAIL_PASSWORD)
                # sending the mail
                smtp.sendmail(Services.EMAIL_SENDER, email, mail.as_string())

            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Email Send Successfully").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(e)
