from __future__ import annotations

import base64
from datetime import datetime

from Cryptodome.Cipher import AES
from passlib.context import CryptContext
from validate_email import validate_email

from scripts.config import Services, Secrets
from scripts.database.mongo.mongo_login import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.utils.security.apply_encrytion_util import create_token


class NormalLogin:
    def __init__(self):
        self.mongo_user = MongoUser()
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.db_user_data = None
        self.db_user_data = None
        self.dt = datetime.now()
        self.time_dt = datetime.now()

    @staticmethod
    def un_pad(s):
        # un_padding the encrypted password
        return s[:-ord(s[len(s) - 1:])]

    def password_decrypt(self, password):
        try:
            # encoding the Key
            key = Secrets.KEY_ENCRYPTION.encode(Services.ENCODING_TYPE)
            # decoding the received password
            enc = base64.b64decode(password)
            # mode for the decryption
            mode = AES.MODE_CBC
            # getting the initialization vector
            iv = enc[:AES.block_size]
            # decoding with AES
            cipher = AES.new(key, mode, iv)
            # decoding the password
            data = cipher.decrypt(enc[AES.block_size:])
            if len(data) == 0:
                raise ValueError("Decrypted data is empty")
            # removing the padding to get the password
            data = self.un_pad(data)
            return data.decode(Services.ENCODING_TYPE)
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def user_data_validation(username, project_id,  password) -> dict | None:
        try:
            # checking for valid username
            if username == "" or username == "user@example.com" or validate_email(username) is not True:
                return {"message": ErrorMessages.ERROR_INVALID_USERNAME, "data": username}
            # checking for valid password
            if password == "" or password == "string":
                return {"message": ErrorMessages.ERROR_INVALID_PASSWORD, "data": password}
            # check if the project id matches
            if Services.PROJECT_ID != project_id:
                return {"message": ErrorMessages.ERROR_INVALID_PROJECT_ID, "data": project_id}
            return None
        except Exception as e:
            logger.exception(e)

    def db_data_validation(self, login_type, username):
        try:
            # fetching the data based on the username
            self.db_user_data = MongoUser().fetch_user_details(username)
            # if the user is not available
            if not self.db_user_data:
                return False, {"message": ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN,
                               "data": {"username": username}}

            # if the user is not registered through normal login
            if self.db_user_data["login_type"] != login_type:
                return False, {"message": ErrorMessages.ERROR_LOGIN_TYPE_INVALID,
                               "data": {"username": username, "Use Login": self.db_user_data["login_type"]}}
            # Check the project id from the request body
            if self.db_user_data["project_id"] != Services.PROJECT_ID:
                return False, {"message": ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN, "data": username}
            # if the user exist
            return None, {"message": True}
        except Exception as e:
            logger.exception(e)

    def db_password_matching(self, login_type, payload, password):
        try:
            # getting the response after checking for the user data in db
            response, message = self.db_data_validation(login_type, payload["username"])
            # if the response is false then an error message is send back
            if response is not None:
                return response, message
            # if the user exists in db then password is matched
            if not self.pwd_context.verify(password, self.db_user_data["password"]):
                return False, {"message": ErrorMessages.ERROR_PASSWORD_MISMATCH,
                               "data": {"username": payload["username"]}}
            # if the password is correct
            return None, {"username": payload["username"], "role": self.db_user_data["user_role"]}
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def generate_cookie_tokens(login_data, request):
        try:
            # creating the access token
            access_token = create_token(
                user_id=login_data["username"],
                ip=request.ip_address,
                project_id=Services.PROJECT_ID,
            )
            # returning the login token
            if access_token:
                return {"user_id": access_token, "token_type": "bearer"}
            else:
                return None
        except Exception as e:
            logger.exception(e)
