from __future__ import annotations

import base64
import datetime

from Cryptodome.Cipher import AES
from passlib.context import CryptContext

from scripts.config import Services
from scripts.database.mongo.mongo_login import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.utils.mongo_default_queries import MongoQueries


class LoginHandlers:
    def __init__(self):
        self.mongo_user = MongoUser()
        self.mongo_queries = MongoQueries()
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.db_user_data = None
        self.db_user_data = None
        self.dt = datetime.datetime.now()
        self.time_dt = datetime.datetime.now()

    @staticmethod
    def un_pad(s):
        # un_padding the encrypted password
        return s[:-ord(s[len(s) - 1:])]

    def password_decrypt(self, password):
        try:
            # encoding the Key
            key = Services.KEY_ENCRYPTION.encode(Services.ENCODING_TYPE)
            # decoding the received password
            enc = base64.b64decode(password)
            # mode for the decryption
            mode = AES.MODE_CBC
            # getting the initialization vector
            iv = enc[:AES.block_size]
            # decoding with AES
            cipher = AES.new(key, mode, iv)
            # decoding the password
            data = cipher.decrypt(enc[AES.block_size:])
            if len(data) == 0:
                raise ValueError("Decrypted data is empty")
            # removing the padding to get the password
            data = self.un_pad(data)
            return data.decode('utf-8')
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def user_data_validation(username, password) -> dict | None:
        try:
            # checking for valid username
            if username == "" or username == "user@example.com":
                return {"message": ErrorMessages.ERROR_INVALID_USERNAME, "data": username}
            # checking for valid password
            if password == "" or password == "string":
                return {"message": ErrorMessages.ERROR_INVALID_PASSWORD, "data": password}
            return None
        except Exception as e:
            logger.exception(e)

    def new_user_login(self, login_data, password):
        # check the domain of the email the user has entered
        if login_data.username.split("@")[-1] == "knowledgelens.com":
            # hash the password
            hashed_password = self.pwd_context.hash(password)
            # Enter the user as a guest user to the db
            if self.mongo_user.insert_user(
                    self.mongo_queries.insert_user_query(
                        login_data,
                        Services.PROJECT_ID,
                        hashed_password,
                        self.time_dt)):
                return True
        return False

    def db_data_validation(self, login_data, password):
        try:
            # fetching the data based on the username
            self.db_user_data = MongoUser().fetch_user_details(login_data.username)
            # if the user is not available
            if not self.db_user_data:
                # if the domain of the user email is knowledge lens create the user as a guest user
                if self.new_user_login(login_data, password):
                    return True, {"message": "new_user", "username": login_data.username, "role": "guest"}
                else:
                    return False, {"message": ErrorMessages.ERROR_INVALID_USERNAME,
                                   "data": {login_data.username, login_data.password}}
            # Check the project id from the request body
            if self.db_user_data["project_id"] != Services.PROJECT_ID or Services.PROJECT_ID != login_data.project_id:
                return False, {"message": ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN, "data": login_data.username}
            # if the user exist
            return None, {"message": True}
        except Exception as e:
            logger.exception(e)

    def db_password_matching(self, login_data, password):
        try:
            # getting the response after checking for the user data in db
            response, message = self.db_data_validation(login_data, password)
            if response and message["message"] == "new_user":
                return None, message
            # if the response is false then an error message is send back
            if response is not None:
                return response, message
            # if the user exists in db then password is matched
            if not self.pwd_context.verify(password, self.db_user_data["password"]):
                return False, {"message": ErrorMessages.ERROR_PASSWORD_MISMATCH,
                               "data": {"username": login_data.username}}
            # if the password is correct
            return None, {"username": login_data.username, "role": self.db_user_data["user_role"]}
        except Exception as e:
            logger.exception(e)
