import base64
from typing import Optional

from Cryptodome.Cipher import AES
from passlib.context import CryptContext

from scripts.config import Secrets, Services
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger


# utility for the password
class EncryptDecryptPassword:
    def __init__(self):
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

    @staticmethod
    def un_pad(s):
        # un_padding the encrypted password
        return s[:-ord(s[len(s) - 1:])]

    def password_decrypt(self, password):
        try:
            # encoding the Key
            key = Secrets.KEY_ENCRYPTION.encode(Services.ENCODING_TYPE)
            # decoding the received password
            enc = base64.b64decode(password)
            # mode for the decryption
            mode = AES.MODE_CBC
            # getting the initialization vector
            iv = enc[:AES.block_size]
            # decoding with AES
            cipher = AES.new(key, mode, iv)
            # decoding the password
            data = cipher.decrypt(enc[AES.block_size:])
            if len(data) == 0:
                raise ValueError("Decrypted data is empty")
            # removing the padding to get the password
            data = self.un_pad(data)
            return data.decode(Services.ENCODING_TYPE)
        except Exception as e:
            logger.error(f'Services Failed with error from password util password decrypt {e}')
            return None

    # encrypting the password
    def password_encrypt(self, password):
        try:
            # decrypting the UI password
            decrypted_password = self.password_decrypt(password)
            # hashing the decrypted password
            if decrypted_password is None:
                return None
            hashed_password = self.pwd_context.hash(decrypted_password.split("\"")[1])
            return hashed_password
        except Exception as e:
            logger.error(f'Services Failed with error from password util password encrypt {e}')
            return None

    def check_password_mismatch(self, new_password, confirm_password, old_password: Optional[str] = None):
        try:
            # decrypting the UI password
            password_decrypted = self.password_decrypt(new_password)
            confirm_decrypted = self.password_decrypt(confirm_password)
            if password_decrypted != confirm_decrypted:
                return None, ErrorMessages.ERROR_MISMATCH_CONFIRM
            # hashing the decrypted password
            if confirm_decrypted is None:
                return None, ErrorMessages.ERROR_PASSWORD_EMPTY
            if old_password is not None:
                return True, confirm_decrypted
            hashed_password = self.pwd_context.hash(confirm_decrypted.split("\"")[1])
            return True, hashed_password
        except Exception as e:
            logger.error(f'Services Failed with error from password util check password mismatch {e}')
            return None, ErrorMessages.ERROR_PASSWORD_FAILED
