import pandas as pd
from pymongo import MongoClient
from datetime import datetime, timedelta
import os
import openpyxl
from openpyxl import load_workbook
from openpyxl.drawing.image import Image
import yaml
from dotenv import load_dotenv
from collections import defaultdict
import smtplib
import requests
import logging

load_dotenv(dotenv_path='.env')


class DailyReportGenerator:

    def __init__(self):
        # Connect to MongoDB
        client = MongoClient(os.environ["MONGO_CLIENT"])
        db = client[os.environ["MONGO_DATABASE"]]
        self.collection = db[os.environ["MONGO_COLLECTION"]]

        # Load configuration from config.yml
        with open('config.yml') as config_file:
            self.config = yaml.safe_load(config_file)
            self.template_file = self.config['template_file']
            self.camera_mappings = {
                item['camera_name']: item['packer_name']
                for item in self.config['cameras']
            }


    def get_shift_name(self, timestamp):

        hour = timestamp.hour
        if 6 <= hour < 14:
            return 'Shift - A'
        elif 14 <= hour < 22:
            return 'Shift - B'
        else:
            return 'Shift - C'
        

    def get_packer_name(self, camera_name):
        return self.camera_mappings.get(camera_name, 'Unknown')    


    def get_count(self, start_time, end_time, camera_name):
        query = {
            'timestamp': {'$gte': start_time, '$lte': end_time},
            'cameraName': camera_name
        }
        documents = self.collection.find(query).sort('timestamp', 1)

        count_difference = 0
        first_count = 0
        last_count = 0  

        for i, document in enumerate(documents):
            if i == 0:
                first_count = int(document['cement_bag_count'])
            last_count = int(document['cement_bag_count'])

        count_difference = last_count - first_count

        return count_difference
    

    def create_excel_report(self):

        report_file = 'daily_report.xlsx' 
        

        current_time = datetime.now()

        # Generate a unique identifier (e.g., timestamp)
        unique_id = current_time.strftime("%Y%m%d%H%M%S%f")

        # Calculate the previous day's date
        previous_day = current_time - timedelta(days=1)
        report_date = previous_day.strftime("%Y-%m-%d")
        report_file = f"daily_report_{report_date}_{unique_id}.xlsx"

        wb = load_workbook(self.template_file)
        sheet = wb.active

        # Set report date
        sheet['F5'] = report_date
        serial_number = 1

        # Define the shifts and their respective time ranges
        shifts = {
            'Shift - A': (datetime(current_time.year, current_time.month, current_time.day -1 , 6, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day -1, 14, 0, 0)),
            'Shift - B': (datetime(current_time.year, current_time.month, current_time.day -1, 14, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day -1, 22, 0, 0)),
            'Shift - C': (datetime(current_time.year, current_time.month, current_time.day- 1, 22, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day, 6, 0, 0))
        }

        # Start filling details from row 8
        row = 7
        total_count = 0  # Initialize total count
        packer_counts = defaultdict(int)  # Initialize packer counts dictionary

        # Get the camera_names and corresponding packer_names from config.yml
        camera_names = [item['camera_name'] for item in self.config['cameras']]
        packer_names = [item['packer_name'] for item in self.config['cameras']]

        for camera_name, packer_name in zip(camera_names, packer_names):
            for shift, (start_time, end_time) in shifts.items():
                count = self.get_count(start_time, end_time, camera_name)

                # Fill in the data in respective cells
                sheet[f'A{row}'] = serial_number
                sheet[f'B{row}'] = end_time.date()
                sheet[f'C{row}'] = packer_name
                sheet[f'D{row}'] = shift
                sheet[f'E{row}'] = count

                total_count += count  # Accumulate the count
                packer_counts[packer_name] += count  # Accumulate the count for the packer

                row += 1
                serial_number += 1

        # Fill the total count in cell E26
        sheet['F25'] = total_count

        # Fill the packer counts in column L
        for packer_name, count in packer_counts.items():
            packer_row = 9 + packer_names.index(packer_name) * 3            
            sheet[f'F{packer_row}'] = count

        # Set print area to include all cells with data
        max_row = sheet.max_row
        max_column = sheet.max_column
        print_area = f"A1:{openpyxl.utils.get_column_letter(max_column)}{max_row}"
        sheet.print_area = print_area

        # Save the report file
        wb.save(report_file)

        # Return the absolute path of the generated report file
        file_path = os.path.abspath(report_file)
       
        return file_path
    

    def save_report_locally(self, filepath):
        # Create the 'daily_reports' folder if it doesn't exist
        folder_path = self.config['folder_path']
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

        # Get the filename from the filepath
        filename = os.path.basename(filepath)

        # Save the report file in the 'daily_reports' folder
        destination = os.path.join(folder_path, filename)
        os.rename(filepath, destination)

        return destination


    def get_receiver_emails(self):
        # Connect to MongoDB and retrieve the email addresses from the collection
        client = MongoClient(os.environ["MONGO_CLIENT"])
        db = client[os.environ["MONGO_DATABASE"]]
        collection = db[os.environ["MONGO_COLLECTION_EMAIL"]]

        emails = []
        documents = collection.find()
        for document in documents:
            emails.append(document["email"])

        return emails


    def send_email_from_ut(self,filename="daily_report.xlsx", filepath=None):
        
        logging.info("Sending email to {}".format(self.get_receiver_emails()))
        payload = dict()
        payload['from_name'] = os.environ["FROM_ADDRESS"]
        payload['receiver_list'] = self.get_receiver_emails()
        payload['subject'] = self.config['email']['subject']
        payload['content'] = self.config['email']['body']
        payload['gateway_id'] = os.environ["GATEWAY"]
        headers = {'authorization': os.environ["AUTHORIZATION"]}
        count = 0
        while count < 3:
            try:
                if filename and filepath:
                    files = [('attachments', (filename, open(filepath, 'rb'), 'application/html'))]
                    response = requests.request("POST", os.environ["URL_WITH_ATTACH"], data=payload, 
                                                headers=headers, files=files, timeout=10)
                    logging.info(f"Response status code for request is: {response.status_code}")
                    if response.status_code == 200:
                        return True
                else:
                    response = requests.request("POST", os.environ["URL_WITHOUT_ATTACH"], json=payload, 
                                                headers=headers, timeout=10)
                    logging.info(f"Response status code for request is: {response.status_code}")
                    if response.status_code == 200:
                        return True
            except Exception as e:
                logging.error(e)
            count += 1
        return False
