import pandas as pd
from pymongo import MongoClient
from datetime import datetime, timedelta
import os
from openpyxl import load_workbook
from openpyxl.drawing.image import Image
import yaml
from dotenv import load_dotenv

load_dotenv(dotenv_path='.env')


class DailyReportGenerator:

    def __init__(self):
        # Connect to MongoDB
        client = MongoClient(os.environ["MONGO_CLIENT"])
        db = client[os.environ["MONGO_DATABASE"]]
        self.collection = db[os.environ["MONGO_COLLECTION"]]

        # Load configuration from config.yml
        with open('config.yml') as config_file:
            config = yaml.safe_load(config_file)
            self.template_file = config['template_file']
            # self.logo_image = config['logo_image']

    def get_shift_name(self, timestamp):
        hour = timestamp.hour
        if 6 <= hour < 14:
            return 'Shift A'
        elif 14 <= hour < 22:
            return 'Shift B'
        else:
            return 'Shift C'

    def map_packer_name(self, camera_name):
        packer_mapping = {
            'jk_4cebd0ec': 'Packer 1',
            'camera_42': 'Packer 2',
            'camera_44': 'Packer 4',
            'camera_45': 'Packer 5',
            'camera_46': 'Packer 6',
            'camera_47': 'Packer 7'
        }
        return packer_mapping.get(camera_name, 'Unknown')

    def get_count(self, start_time, end_time, camera_name):
        query = {
            'timestamp': {'$gte': start_time, '$lte': end_time},
            'cameraName': camera_name
        }
        documents = self.collection.find(query).sort('timestamp', 1)
        first_document = None
        last_document = None

        for document in documents:
            if first_document is None:
                first_document = document
            last_document = document

        if first_document is None or last_document is None:
            return 0

        count_difference = int(last_document['cement_bag_count']) - int(first_document['cement_bag_count'])
        return count_difference

    def create_excel_report(self):

        report_file = 'daily_report.xlsx'

        current_time = datetime.now()
        report_date = current_time.strftime("%Y-%m-%d")
        report_file = f"daily_report_{report_date}.xlsx"

        start_time = datetime(current_time.year, current_time.month, current_time.day - 1, 6, 0, 0)
        end_time = datetime(current_time.year, current_time.month, current_time.day, 6, 0, 0)

        wb = load_workbook(self.template_file)
        sheet = wb.active

        # Add logos
        # logo_image = Image(self.logo_image)
        # sheet.add_image(logo_image, 'A1')
        # sheet.merge_cells('A1:C3')

        # Set report date
        sheet['L6'] = report_date
        serial_number = 1

        # Start filling details from row 8
        row = 8
        for camera_name in ['jk_4cebd0ec', 'camera_42', 'camera_44', 'camera_45', 'camera_46', 'camera_47']:
            shift_name = self.get_shift_name(start_time)
            packer_name = self.map_packer_name(camera_name)
            count = self.get_count(start_time, end_time, camera_name)

            # Fill in the data in respective cells
            sheet[f'A{row}'] = serial_number
            sheet[f'B{row}'] = start_time.date()
            sheet[f'C{row}'] = shift_name
            sheet[f'D{row}'] = packer_name
            sheet[f'E{row}'] = count

            row += 1
            serial_number += 1

        # Save the report file
        wb.save(report_file)

        # Return the absolute path of the generated report file
        file_path = os.path.abspath(report_file)
        return file_path

# Create an instance of DailyReportGenerator
report_generator = DailyReportGenerator()

# Generate the daily report
report_path = report_generator.create_excel_report()

# Print the path of the generated report
print(f"Generated Daily Report: {report_path}")
